﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Threading.Tasks;
using System.Web.Http;
using System.Web.Http.Description;
using BingMapsRESTToolkit;
using Microsoft.Web.Http;
using VA.PPMS.Context;
using Newtonsoft.Json;
using System.Net.Http.Formatting;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using Microsoft.AspNet.OData;
using Microsoft.AspNet.OData.Routing;
using PpmsDataWebService.Models;
using PpmsDataWebService.ModelsEnumTypes;
using PpmsDataWebService.Helpers;
using Microsoft.Xrm.Sdk.Messages;

namespace PpmsDataService.Controllers
{

    //[Authorize]
    [ApiVersion("1.0")]
    public class VAFacilityLocatorController : ODataController
    {    
        
        [ResponseType(typeof(VAFacilityLocatorResult))]
        [ODataRoute("VAFacilityLocator")]
        public async Task<HttpResponseMessage> GetVAFacilityLocator([FromODataUri] string address, [FromODataUri] int radius, [FromODataUri] int facilityType1, [FromODataUri] int facilityType2, [FromODataUri] int facilityType3, [FromODataUri] int facilityType4)
        {

            //Default Value for maxResults
            var maxResults = 2500; 

            var _service = await PpmsContextHelper.GetProxy();

            using (var context = new PpmsContext(await PpmsContextHelper.GetProxy()))
            {
                string key = ConfigurationManager.AppSettings["BingMapsKey"];

                //Geocode the Starting Address first
                var request = new GeocodeRequest()
                {
                    Query = address,
                    IncludeIso2 = true,
                    IncludeNeighborhood = true,
                    MaxResults = 25,
                    BingMapsKey = key
                };

                //Process the request by using the ServiceManager.
                var response = await ServiceManager.GetResponseAsync(request);

                if (response != null &&
                    response.ResourceSets != null &&
                    response.ResourceSets.Length > 0 &&
                    response.ResourceSets[0].Resources != null &&
                    response.ResourceSets[0].Resources.Length > 0)
                {
                    var result = response.ResourceSets[0].Resources[0] as Location;
                    if (result != null)
                    {
                        double startLatitude = result.Point.Coordinates[0];
                        double startLongitude = result.Point.Coordinates[1];

                        //Convert to Meters for formula.
                        double distanceCalc = radius * 1609.34;

                        //Only need the Northwest and Southeast coordinates to determine Bounds
                        var northWestBoundCoord = DistanceCalc.getBounds(startLatitude, startLongitude, 315, distanceCalc);
                        decimal northWestBoundLong = (decimal)northWestBoundCoord.Longitude;
                        decimal northWestBoundLat = (decimal)northWestBoundCoord.Latitude;

                        var southEastBoundCoord = DistanceCalc.getBounds(startLatitude, startLongitude, 135, distanceCalc);
                        decimal southEastBoundLong = (decimal)southEastBoundCoord.Longitude;
                        decimal southEastBoundLat = (decimal)southEastBoundCoord.Latitude;

                        //Build the FetchXml
                        string fetchXml =
                            String.Format(@"<fetch version='1.0' output-format='xml-platform' mapping='logical' distinct='false'>
                            <entity name='ppms_caresite'> 
                            <attribute name='ppms_name' /> 
                            <attribute name='ppms_facilitytype' />
                            <attribute name='ppms_vacaresite' /> 
                            <attribute name='ppms_caresiteid' /> 
                            <attribute name='ppms_address_postalcode' /> 
                            <attribute name='ppms_address_line1' />
                            <attribute name='ppms_stationnumber' />
                            <attribute name='ppms_stationname' />
                            <attribute name='ppms_statename' />
                            <attribute name='ppms_mainsitephone' />
                            <attribute name='ppms_longitude' />
                            <attribute name='ppms_latitude' />
                            <attribute name='ppms_address_city' />
                            <filter type='and'>
                            <condition attribute='ppms_geocoded' operator='eq' value='1' />
                            <condition attribute='ppms_addressvalidated' operator='eq' value='1' />
                            <condition attribute='statecode' operator='eq' value='0' />
                            <condition attribute='statuscode' operator='eq' value='1' />
                            <condition attribute='ppms_vacaresite' operator='eq' value='1' />
                            <condition attribute='ppms_latitude' operator='not-null' />
                            <condition attribute='ppms_longitude' operator='not-null' />
                            <condition attribute='ppms_latitude' operator='lt' value='{0}'/>
                            <condition attribute='ppms_latitude' operator='gt' value='{1}'/>
                            <condition attribute='ppms_longitude' operator='gt' value='{2}'/>
                            <condition attribute='ppms_longitude' operator='lt' value='{3}'/>
                            </filter>", northWestBoundLat, southEastBoundLat, northWestBoundLong, southEastBoundLong);

                        //Check the Facility Type Params and look up the associated Optionset Value.
                        if (facilityType1 != 0 || facilityType2 != 0 || facilityType3 != 0 || facilityType4 != 0)
                        {
                            fetchXml += "<filter type='or'>";
                            if (facilityType1 != 0 && facilityType1 <= 5) {
                                var facilityType = VAFacilityIds.GetFacility(facilityType1);
                                fetchXml += String.Format("<condition attribute='ppms_facilitytype' operator='eq' value='{0}'/>", facilityType.Value);
                            }
                            if (facilityType2 != 0 && facilityType2 <= 5)
                            {
                                var facilityType = VAFacilityIds.GetFacility(facilityType2);
                                fetchXml += String.Format("<condition attribute='ppms_facilitytype' operator='eq' value='{0}'/>", facilityType.Value);
                            }
                            if (facilityType3 != 0 && facilityType3 <= 5)
                            {
                                var facilityType = VAFacilityIds.GetFacility(facilityType3);
                                fetchXml += String.Format("<condition attribute='ppms_facilitytype' operator='eq' value='{0}'/>", facilityType.Value);
                            }
                            if (facilityType4 != 0 && facilityType4 <=5)
                            {
                                var facilityType = VAFacilityIds.GetFacility(facilityType4);
                                fetchXml += String.Format("<condition attribute='ppms_facilitytype' operator='eq' value='{0}'/>", facilityType.Value);
                            }                           
                            fetchXml += "</filter>";
                        }

                        fetchXml += "</entity ></fetch >";

                        // Set the number of records per page to retrieve.
                        int fetchCount = 5000;
                        // Initialize the page number.
                        int pageNumber = 1;
                        // Specify the current paging cookie. For retrieving the first page, 
                        // pagingCookie should be null.
                        string pagingCookie = null;

                        List<ppms_caresite> vaFacilityList = new List<ppms_caresite>();

                        while (true)
                        {
                            // Build fetchXml string with the placeholders.
                            string xml = FetchHelper.CreateXml(fetchXml, pagingCookie, pageNumber, fetchCount);

                            // Excute the fetch query and get the xml result.
                            RetrieveMultipleRequest fetchRequest = new RetrieveMultipleRequest
                            {
                                Query = new FetchExpression(xml)
                            };

                            EntityCollection returnCollection = ((RetrieveMultipleResponse)_service.Execute(fetchRequest)).EntityCollection;

                            //Add to the Provider list
                            vaFacilityList.AddRange(returnCollection.Entities.Select(e => e.ToEntity<ppms_caresite>()).ToList());

                            // Check for morerecords, if it returns 1.
                            if (returnCollection.MoreRecords)
                            {
                                // Increment the page number to retrieve the next page.
                                pageNumber++;
                                // Set the paging cookie to the paging cookie returned from current results.                            
                                pagingCookie = returnCollection.PagingCookie;
                            }
                            else
                            {
                                // If no more records in the result nodes, exit the loop.
                                break;
                            }
                        }

                        if (vaFacilityList.Count > 2500)
                        {
                            var vaFacilityListDistances = vaFacilityList
                                .Select(vaf => new VaFacilityInRadius
                                {
                                    VaFacility = vaf,
                                    Distance = DistanceCalc.GetDistance(startLatitude, startLongitude
                                 , Convert.ToDouble(vaf.ppms_latitude), Convert.ToDouble(vaf.ppms_longitude))
                                }).AsEnumerable();

                            //Add Providers to new list which meet the Radius criteria and Order by Distance. 
                            vaFacilityList = vaFacilityListDistances.Where(vaf => vaf.Distance <= radius).OrderBy(vaf => vaf.Distance).Take(maxResults).Select(x => x.VaFacility).ToList();
                        }

                        if (vaFacilityList.Count > 0)
                        {
                            var vaFacilityLocatorList = new List<VAFacilityLocatorResult>();

                            using (var client = new HttpClient())
                            {
                                var origins = new List<PpmsDataWebService.Models.DistanceMatrixRequest.Origin>();
                                var origin = new PpmsDataWebService.Models.DistanceMatrixRequest.Origin() { latitude = startLatitude, longitude = startLongitude };
                                origins.Add(origin);
                                var destinations = new List<PpmsDataWebService.Models.DistanceMatrixRequest.Destination>();
                                foreach (ppms_caresite vaFacility in vaFacilityList)
                                {
                                    double latitude = Convert.ToDouble(vaFacility.ppms_latitude);
                                    double longitude = Convert.ToDouble(vaFacility.ppms_longitude);
                                    var destination = new PpmsDataWebService.Models.DistanceMatrixRequest.Destination() { latitude = latitude, longitude = longitude };
                                    destinations.Add(destination);
                                }

                                //Create Request Content Objecet
                                var distanceMatrixRequestContent = new PpmsDataWebService.Models.DistanceMatrixRequest.RootObject()
                                {
                                    origins = origins,
                                    destinations = destinations,
                                    timeUnits = "minutes",
                                    travelMode = "driving",
                                    distanceUnits = "mile"
                                };
                                //Can only optimize with traffic if total results is 10 or less. 
                                if (vaFacilityList.Count <= 10) { distanceMatrixRequestContent.StartTime = DateTime.Now.ToString(); }

                                //Serialize Content to Json
                                //var content = JsonConvert.SerializeObject(distanceMatrixRequestContent);
                                MediaTypeFormatter jsonFormatter = new JsonMediaTypeFormatter();
                                HttpContent httpContent = new ObjectContent<PpmsDataWebService.Models.DistanceMatrixRequest.RootObject>(distanceMatrixRequestContent, jsonFormatter);
                                var distanceMatrixPostUrl = String.Format("https://dev.virtualearth.net/REST/v1/Routes/DistanceMatrix?key={0}", key);
                                httpContent.Headers.ContentType.MediaType = ("application/json");
                                var matrixRequest = new HttpRequestMessage()
                                {
                                    RequestUri = new Uri(distanceMatrixPostUrl),
                                    Method = HttpMethod.Post,
                                    Content = httpContent
                                };
                                //matrixRequest.Headers.Add("Content-Type", "application/json");
                                HttpResponseMessage responseContent = client.SendAsync(matrixRequest).Result;

                                var json = responseContent.Content.ReadAsStringAsync().Result;

                                var matrixResponse = JsonConvert.DeserializeObject<PpmsDataWebService.Models.DistanceMatrixResponse.RootObject>(json);
                                if (matrixResponse != null &&
                                matrixResponse.resourceSets != null &&
                                matrixResponse.resourceSets.Count > 0 &&
                                matrixResponse.resourceSets[0].resources != null &&
                                matrixResponse.resourceSets[0].resources.Count > 0)
                                {
                                    var travelTimesDistances = matrixResponse.resourceSets[0].resources[0].results;

                                    for (var i = 0; i < travelTimesDistances.Count; i++)
                                    {

                                        //Only do mapping if the VA Facility falls within Radius
                                        if (travelTimesDistances[i].travelDistance <= radius)
                                        {
                                            var vaFacilityLocatorResult = new VAFacilityLocatorResult();
                                            vaFacilityLocatorResult.Miles = travelTimesDistances[i].travelDistance;
                                            vaFacilityLocatorResult.Minutes = travelTimesDistances[i].travelDuration;
                                            vaFacilityLocatorResult.Latitude = Convert.ToDouble(vaFacilityList[i].ppms_latitude);
                                            vaFacilityLocatorResult.Longitude = Convert.ToDouble(vaFacilityList[i].ppms_longitude);
                                            vaFacilityLocatorResult.FacilityName = vaFacilityList[i].ppms_name;
                                            vaFacilityLocatorResult.Address = vaFacilityList[i].ppms_address_line1;
                                            vaFacilityLocatorResult.City = vaFacilityList[i].ppms_address_city;
                                            vaFacilityLocatorResult.State = vaFacilityList[i].ppms_statename;
                                            vaFacilityLocatorResult.ZipCode = vaFacilityList[i].ppms_address_postalcode;
                                            vaFacilityLocatorResult.StationName = vaFacilityList[i].ppms_stationname;
                                            vaFacilityLocatorResult.StationNumber = vaFacilityList[i].ppms_stationnumber;
                                            vaFacilityLocatorResult.MainSitePhone = vaFacilityList[i].ppms_mainsitephone;

                                            //Facility Type Switch
                                            if (vaFacilityList[i].ppms_FacilityType != null) {
                                                switch (vaFacilityList[i].ppms_FacilityType.Value)
                                                {
                                                    case (int)ppms_sitetype.VAMC:
                                                        vaFacilityLocatorResult.FacilityType = VAFacilityType.VAMC;
                                                        break;
                                                    case (int)ppms_sitetype.PCCBOC:
                                                        vaFacilityLocatorResult.FacilityType = VAFacilityType.PC_CBOC;
                                                        break;
                                                    case (int)ppms_sitetype.OOS:
                                                        vaFacilityLocatorResult.FacilityType = VAFacilityType.OOS;
                                                        break;
                                                    case (int)ppms_sitetype.MSCBOC:
                                                        vaFacilityLocatorResult.FacilityType = VAFacilityType.MS_CBOC;
                                                        break;
                                                    case (int)ppms_sitetype.HCC:
                                                        vaFacilityLocatorResult.FacilityType = VAFacilityType.HCC;
                                                        break;
                                                    default:
                                                        break;
                                                }
                                            }

                                            vaFacilityLocatorList.Add(vaFacilityLocatorResult);
                                        }
                                    }
                                }
                            }
                            //Sort the Results on Distance ascending.
                            vaFacilityLocatorList.Sort((x, y) => x.Miles.CompareTo(y.Miles));
                            //Return the Results
                            return Request.CreateResponse(vaFacilityLocatorList);
                        }
                    }
                }
            }
            var message = string.Format("No VA Facilities found based on Address, Radius and Filter Criteria given");
            HttpError err = new HttpError(message);
            return Request.CreateErrorResponse(HttpStatusCode.NotFound, err);
        }


        public static class DistanceCalc
        {
            public static bool IsInRadius(double lat1, double lon1, double lat2, double lon2, double radius)
            {
                var R = 6372.8; // In kilometers
                var dLat = toRadians(lat2 - lat1);
                var dLon = toRadians(lon2 - lon1);
                lat1 = toRadians(lat1);
                lat2 = toRadians(lat2);

                var a = Math.Sin(dLat / 2) * Math.Sin(dLat / 2) + Math.Sin(dLon / 2) * Math.Sin(dLon / 2) * Math.Cos(lat1) * Math.Cos(lat2);
                var c = 2 * Math.Asin(Math.Sqrt(a));
                var distance = R * 2 * Math.Asin(Math.Sqrt(a));
                if (distance <= radius)
                {
                    return true;
                }
                return false;
            }

            public static double GetDistance(double lat1, double lon1, double lat2, double lon2)
            {
                var R = 6372.8; // In kilometers
                var dLat = toRadians(lat2 - lat1);
                var dLon = toRadians(lon2 - lon1);
                lat1 = toRadians(lat1);
                lat2 = toRadians(lat2);
                var a = Math.Sin(dLat / 2) * Math.Sin(dLat / 2) + Math.Sin(dLon / 2) * Math.Sin(dLon / 2) * Math.Cos(lat1) * Math.Cos(lat2);
                var c = 2 * Math.Asin(Math.Sqrt(a));
                var distance = R * 2 * Math.Asin(Math.Sqrt(a));
                return distance;
            }

            

            public static double toRadians(double angle)
            {
                return Math.PI * angle / 180.0;
            }


            public static Coordinate getBounds(double lat1, double lon1, double brng, double dist)
            {
                var a = 6378137;
                var b = 6356752.3142;
                var f = 1 / 298.257223563; // WGS-84 ellipsiod
                var s = dist;
                var alpha1 = toRad(brng);
                var sinAlpha1 = Math.Sin(alpha1);
                var cosAlpha1 = Math.Cos(alpha1);
                var tanU1 = (1 - f) * Math.Tan(toRad(lat1));
                var cosU1 = 1 / Math.Sqrt((1 + tanU1 * tanU1));
                var sinU1 = tanU1 * cosU1;
                var sigma1 = Math.Atan2(tanU1, cosAlpha1);
                var sinAlpha = cosU1 * sinAlpha1;
                var cosSqAlpha = 1 - sinAlpha * sinAlpha;
                var uSq = cosSqAlpha * (a * a - b * b) / (b * b);
                var A = 1 + uSq / 16384 * (4096 + uSq * (-768 + uSq * (320 - 175 * uSq)));
                var B = uSq / 1024 * (256 + uSq * (-128 + uSq * (74 - 47 * uSq)));
                var sigma = s / (b * A);
                var sigmaP = 2 * Math.PI;
                double sinSigma = 0;
                double cosSigma = 0;
                double cos2SigmaM = 0;
                while (Math.Abs(sigma - sigmaP) > 1e-12)
                {
                    cos2SigmaM = Math.Cos(2 * sigma1 + sigma);
                    sinSigma = Math.Sin(sigma);
                    cosSigma = Math.Cos(sigma);
                    var deltaSigma = B * sinSigma * (cos2SigmaM + B / 4 * (cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM) - B / 6 * cos2SigmaM * (-3 + 4 * sinSigma * sinSigma) * (-3 + 4 * cos2SigmaM * cos2SigmaM)));
                    sigmaP = sigma;
                    sigma = s / (b * A) + deltaSigma;
                };
                var tmp = sinU1 * sinSigma - cosU1 * cosSigma * cosAlpha1;
                var lat2 = Math.Atan2(sinU1 * cosSigma + cosU1 * sinSigma * cosAlpha1, (1 - f) * Math.Sqrt(sinAlpha * sinAlpha + tmp * tmp));
                var lambda = Math.Atan2(sinSigma * sinAlpha1, cosU1 * cosSigma - sinU1 * sinSigma * cosAlpha1);
                var C = f / 16 * cosSqAlpha * (4 + f * (4 - 3 * cosSqAlpha));
                var L = lambda - (1 - C) * f * sinAlpha * (sigma + C * sinSigma * (cos2SigmaM + C * cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM)));
                var revAz = Math.Atan2(sinAlpha, -tmp); // final bearing

                var latitude = toDeg(lat2);
                var longitude = lon1 + toDeg(L);
                return new Coordinate(latitude, longitude);
            }
            public static double toRad(double n)
            {
                return n * Math.PI / 180;
            }
            public static double toDeg(double n)
            {
                return n * 180 / Math.PI;
            }
        }

        public class VaFacilityInRadius
        {
            public ppms_caresite VaFacility { get; set; }
            public double Distance { get; set; }
        }
       
    }
}
